
#include <usb.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <fcntl.h>
#include "jtag.h"

enum {
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__SET_MASKS,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__SET_STATUS,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_INIT1,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_INIT2,
  USB_VENDOR_REQUEST__CONFIGURE_XLATORX__JTAG_DEINIT
};

#define TDI 1
#define TMS 2

#define CMD_READ 0x0011
#define CMD_WP   0x0012
#define CMD_WPL  0x0022
#define CMD_EWP  0x0032
#define CMD_EWPL 0x0042
#define CMD_EA   0x0013
#define CMD_SLB  0x0014
#define CMD_CLB  0x0024
#define CMD_GLB  0x0015
#define CMD_SGPB 0x0034
#define CMD_CGPB 0x0044
#define CMD_GGPB 0x0025
#define CMD_SSE  0x0054
#define CMD_GSE  0x0035
#define CMD_WRAM 0x001F
#define CMD_SEFC 0x0016
#define CMD_GVE  0x001E

jtag_commands cmd_buf;
int verbose;


unsigned long read_dccr(struct usb_dev_handle* handle) {
  unsigned char send_buf[5], recv_buf[5];
  int ret;
  send_buf[0] = 0;
  send_buf[1] = 0;
  send_buf[2] = 0;
  send_buf[3] = 0;
  send_buf[4] = 0x04;

  jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 0);
  jtag_EXIT_DR_to_SHIFT_DR_via_UPDATE_DR(&cmd_buf);
  jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 1);
  jtag_EXIT_DR_to_SHIFT_DR_direct(&cmd_buf);
  ret = jtag_run_commands(&cmd_buf);
  if( ret < 0 ) {
    fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
    abort();
  }
  if( (ret = jtag_get_response(&cmd_buf, recv_buf, 38)) != 38 ) {
    fprintf(stderr, "jtag_run_commands failed (%d != 38) (%d)\n", ret, cmd_buf.resp_bits);
    abort();
  }

  if( verbose )
    printf("read DCCR: %02x%02x%02x%02x%02x\n", recv_buf[4], recv_buf[3], recv_buf[2], recv_buf[1], recv_buf[0]);
//  fflush(stdout);
  return ((unsigned long*)recv_buf)[0];
}

unsigned long read_dccr_until(struct usb_dev_handle* handle, unsigned long value, unsigned long mask) {
  unsigned char send_buf[5], recv_buf[5];
  int ret;
  send_buf[0] = 0;
  send_buf[1] = 0;
  send_buf[2] = 0;
  send_buf[3] = 0;
  send_buf[4] = 0x04;

  jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 0);
  jtag_EXIT_DR_to_SHIFT_DR_via_UPDATE_DR(&cmd_buf);
  jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 1);
  jtag_EXIT_DR_to_SHIFT_DR_direct(&cmd_buf);
  ret = jtag_run_commands(&cmd_buf);
  if( ret < 0 ) {
    fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
    abort();
  }
  if( (ret = jtag_get_response(&cmd_buf, recv_buf, 38)) != 38 ) {
    fprintf(stderr, "jtag_run_commands failed (%d != 38) (%d)\n", ret, cmd_buf.resp_bits);
    abort();
  }
  while( (((unsigned long*)recv_buf)[0]&mask) != value ) {
    printf("waiting (%08x)...\n", ((unsigned long*)recv_buf)[0]);
    jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 1);
    jtag_EXIT_DR_to_SHIFT_DR_direct(&cmd_buf);
    ret = jtag_run_commands(&cmd_buf);
    if( ret < 0 ) {
      fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
      abort();
    }
    if( (ret = jtag_get_response(&cmd_buf, recv_buf, 38)) != 38 ) {
      fprintf(stderr, "jtag_run_commands failed (%d != 38) (%d)\n", ret, cmd_buf.resp_bits);
      abort();
    }
  }

  if( verbose )
    printf("read DCCR: %02x%02x%02x%02x%02x\n", recv_buf[4], recv_buf[3], recv_buf[2], recv_buf[1], recv_buf[0]);
//  fflush(stdout);
  return ((unsigned long*)recv_buf)[0];
}

unsigned long read_data(struct usb_dev_handle* handle) {
  unsigned char send_buf[5], recv_buf[5];
  int ret;
  send_buf[0] = 0;
  send_buf[1] = 0;
  send_buf[2] = 0;
  send_buf[3] = 0;
  send_buf[4] = 0x05;

  jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 0);
  jtag_EXIT_DR_to_SHIFT_DR_via_UPDATE_DR(&cmd_buf);
  jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 1);
  jtag_EXIT_DR_to_SHIFT_DR_direct(&cmd_buf);
  ret = jtag_run_commands(&cmd_buf);
  if( ret < 0 ) {
    fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
    abort();
  }
  if( (ret = jtag_get_response(&cmd_buf, recv_buf, 38)) != 38 ) {
    fprintf(stderr, "jtag_run_commands failed (%d != 38) (%d)\n", ret, cmd_buf.resp_bits);
    abort();
  }

  if( verbose )
    printf("read DCDR: %02x%02x%02x%02x%02x\n", recv_buf[4], recv_buf[3], recv_buf[2], recv_buf[1], recv_buf[0]);
//  fflush(stdout);
  return ((unsigned long*)recv_buf)[0];
}

void write_data(struct usb_dev_handle* handle, unsigned long data) {
  unsigned char send_buf[5];
  int ret;
  ((unsigned long*)send_buf)[0] = data;
  send_buf[4] = 0x25;
  if( verbose )
    printf("write DCDR: %02x%02x%02x%02x%02x\n", send_buf[4], send_buf[3], send_buf[2], send_buf[1], send_buf[0]);
//  fflush(stdout);

  jtag_shift_bits(&cmd_buf, send_buf, 38, 1, 0);
  jtag_EXIT_DR_to_SHIFT_DR_via_UPDATE_DR(&cmd_buf);
  ret = jtag_run_commands(&cmd_buf);
  if( ret < 0 ) {
    fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
    abort();
  }
}

void write_handshake(struct usb_dev_handle* handle) {
/*
  while( read_dccr(handle)&1 ) {
    printf("Waiting for write handshake...\n");
  }
*/
  read_dccr_until(handle, 0, 1);
}

void read_handshake(struct usb_dev_handle* handle) {
/*
  unsigned long temp;
  while( !((temp = read_dccr(handle))&2) ) {
    printf("Waiting for read handshake...\n");
  }
*/
  read_dccr_until(handle, 2, 2);
}

unsigned long get_chain_length(struct usb_dev_handle* handle) {
  unsigned long i;
  int ret;

  jtag_reset_TAP(&cmd_buf);
  jtag_RESET_to_IDLE(&cmd_buf);
  jtag_IDLE_to_SHIFT_IR(&cmd_buf);
  jtag_shift_ones(&cmd_buf, 255, 1, 0);
  jtag_EXIT_IR_to_SHIFT_DR_via_UPDATE_IR(&cmd_buf);
  jtag_shift_zeros(&cmd_buf, 255, 0, 0);
  ret = jtag_run_commands(&cmd_buf);
  if( ret < 0 ) {
    fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
    abort();
  }

  for( i = 0; i < 100; ++i ) {
    unsigned char resp;

    jtag_shift_bits_dir(&cmd_buf, 0x1, 1, 0, 1);
    ret = jtag_run_commands(&cmd_buf);
    if( ret < 0 ) {
      fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
      abort();
    }
    ret = jtag_get_response(&cmd_buf, &resp, 1);
    if( ret == 1 ) {
      if( resp&1 )
        return i;
    } else {
      return 0;
    }
  }
}

unsigned long get_device_id(struct usb_dev_handle* handle, unsigned long which) {
  unsigned char zero_buf[4] = { 0, 0, 0, 0 }, id_buf[4];
  int ret;

  jtag_reset_TAP(&cmd_buf);
  jtag_RESET_to_IDLE(&cmd_buf);
  jtag_IDLE_to_SHIFT_DR(&cmd_buf);

  jtag_shift_zeros(&cmd_buf, 32, 1, 1);
  ret = jtag_run_commands(&cmd_buf);
  if( ret < 0 ) {
    fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
    abort();
  }
  if( jtag_get_response(&cmd_buf, id_buf, 32) == 32 )
    return ((unsigned long*)id_buf)[0];
  else
    return 0;
}

void init_serial_programming_mode(struct usb_dev_handle* handle) {
  int ret;

  jtag_reset_TAP(&cmd_buf);
  jtag_RESET_to_IDLE(&cmd_buf);
  jtag_IDLE_to_SHIFT_IR(&cmd_buf);
  jtag_shift_bits_dir(&cmd_buf, 0x2, 4, 1, 0);
  jtag_EXIT_IR_to_SHIFT_DR_via_UPDATE_IR(&cmd_buf);
  jtag_shift_bits_dir(&cmd_buf, 0x2, 4, 1, 0);
  jtag_EXIT_DR_to_SHIFT_IR_via_UPDATE_DR(&cmd_buf);
  jtag_shift_bits_dir(&cmd_buf, 0xc, 4, 1, 0);
  jtag_EXIT_IR_to_SHIFT_DR_via_UPDATE_IR(&cmd_buf);

  ret = jtag_run_commands(&cmd_buf);
  if( ret < 0 ) {
    fprintf(stderr, "jtag_run_commands failed (%d)\n", ret);
    abort();
  }
}

void sp_mode_read_flash(struct usb_dev_handle* handle, unsigned long from_addr, unsigned char* to_here, unsigned long num_words) {
  unsigned long i = 0;
  if( verbose )
    printf("Executing READ\n");
  write_handshake(handle);
  write_data(handle, CMD_READ|(num_words<<16));
  write_handshake(handle);
  write_data(handle, from_addr);
  while( num_words ) {
    printf("%08x/%08x (%.2lf%%)\r", i, i+num_words, i*100.0/(i+num_words));
    fflush(stdout);
    read_handshake(handle);
    ((unsigned long*)to_here)[0] = read_data(handle);
    to_here += 4;
    --num_words;
    ++i;
  }
  printf("\r                                  \r");
    fflush(stdout);
}

void sp_mode_write_flash(struct usb_dev_handle* handle, unsigned long to_addr, const unsigned char* from_here, unsigned long num_words, unsigned char erase, unsigned char lock, unsigned long pause) {
  unsigned long i = 0, count, orig_count;
  if( verbose )
    if( lock )
      if( erase )
        printf("Executing EWPL\n");
      else
        printf("Executing WPL\n");
    else
      if( erase )
        printf("Executing EWP\n");
      else
        printf("Executing WP\n");
  while( num_words ) {
    count = num_words;
    if( (to_addr&~0xFF)+num_words*4 != (to_addr&~0xFF) )
      count = 64-((to_addr&0xFF)>>2);
    orig_count = count;
//    printf("num_words = %d, count = %d\n", num_words, count);
    write_handshake(handle);
    write_data(handle, (lock ? (erase ? CMD_EWPL : CMD_WPL) : (erase ? CMD_EWP : CMD_WP))|(count<<16));
    write_handshake(handle);
    write_data(handle, to_addr);
    while( count ) {
      printf("%08x/%08x (%.2lf%%)\r", i, i+num_words, i*100.0/(i+num_words));
      fflush(stdout);
      write_handshake(handle);
      write_data(handle, ((unsigned long*)from_here)[0]);
      from_here += 4;
      --count;
      --num_words;
      ++i;
    }
    to_addr += orig_count*4;
    if( num_words && pause )
      usleep(pause*1000);
  }
  printf("\r                                  \r");
    fflush(stdout);
}

void sp_mode_safe_write_flash(struct usb_dev_handle* handle, unsigned long to_addr, const unsigned char* from_here, unsigned long num_words, unsigned char erase, unsigned char lock, unsigned long pause) {
  unsigned long i = 0, count, orig_count, datum;
  unsigned char failed;
  if( verbose )
    if( lock )
      if( erase )
        printf("Executing EWPL\n");
      else
        printf("Executing WPL\n");
    else
      if( erase )
        printf("Executing EWP\n");
      else
        printf("Executing WP\n");
  while( num_words ) {
    count = num_words;
    if( (to_addr&~0xFF)+num_words*4 != (to_addr&~0xFF) )
      count = 64-((to_addr&0xFF)>>2);
    orig_count = count;
//    printf("num_words = %d, count = %d\n", num_words, count);
    write_handshake(handle);
    write_data(handle, (lock ? (erase ? CMD_EWPL : CMD_WPL) : (erase ? CMD_EWP : CMD_WP))|(count<<16));
    write_handshake(handle);
    write_data(handle, to_addr);

    while( count ) {
      printf("[WRITE]  %08x/%08x (%.2lf%%)\r", i, i+num_words+count-orig_count, i*100.0/(i+num_words+count-orig_count));
      fflush(stdout);
      write_handshake(handle);
      write_data(handle, ((unsigned long*)from_here)[0]);
      from_here += 4;
      --count;
      ++i;
    }

    // verify
    if( num_words && pause )
      usleep(pause*1000);
    count = orig_count;
    from_here -= count*4;
    i -= count;
    write_handshake(handle);
    write_data(handle, CMD_READ|(count<<16));
    write_handshake(handle);
    write_data(handle, to_addr);
    failed = 0;
    while( count ) {
      printf("[VERIFY] %08x/%08x (%.2lf%%)\r", i, i+num_words+count-orig_count, i*100.0/(i+num_words+count-orig_count));
      fflush(stdout);
      read_handshake(handle);
      datum = read_data(handle);
      if( datum !=  ((unsigned long*)from_here)[0] )
        failed = 1;
      from_here += 4;
      --count;
      ++i;
    }

    if( failed ) {
      from_here -= orig_count*4;
      i -= orig_count;
      printf("Retrying address %08x...             \n", i);
      continue;
    }

    to_addr += orig_count*4;
    num_words -= orig_count;
  }
  printf("\r                                  \r");
    fflush(stdout);
}

unsigned long sp_mode_full_erase(struct usb_dev_handle* handle) {
  if( verbose )
    printf("Executing EA\n");
  write_handshake(handle);
  write_data(handle, CMD_EA);
}

unsigned long sp_mode_get_lock_bits(struct usb_dev_handle* handle) {
  if( verbose )
    printf("Executing GLB\n");
  write_handshake(handle);
  write_data(handle, CMD_GLB);
  read_handshake(handle);
  return read_data(handle);
}

void sp_mode_set_lock_bits(struct usb_dev_handle* handle, unsigned long which) {
  if( verbose )
    printf("Executing SLB\n");
  write_handshake(handle);
  write_data(handle, CMD_SLB);
  write_handshake(handle);
  write_data(handle, which);
}

void sp_mode_clear_lock_bits(struct usb_dev_handle* handle, unsigned long which) {
  if( verbose )
    printf("Executing CLB\n");
  write_handshake(handle);
  write_data(handle, CMD_CLB);
  write_handshake(handle);
  write_data(handle, which);
}

unsigned long sp_mode_get_gp_nvm_bits(struct usb_dev_handle* handle) {
  if( verbose )
    printf("Executing CGPB\n");
  write_handshake(handle);
  write_data(handle, CMD_CGPB);
  write_handshake(handle);
  return read_data(handle);
}

void sp_mode_set_gp_nvm_bits(struct usb_dev_handle* handle, unsigned long bits) {
  if( verbose )
    printf("Executing SGPB\n");
  write_handshake(handle);
  write_data(handle, CMD_SGPB);
  write_handshake(handle);
  write_data(handle, bits);
}

void sp_mode_clear_gp_nvm_bits(struct usb_dev_handle* handle, unsigned long bits) {
  if( verbose )
    printf("Executing CGPB\n");
  write_handshake(handle);
  write_data(handle, CMD_CGPB);
  write_handshake(handle);
  write_data(handle, bits);
}

void sp_mode_set_security_bits(struct usb_dev_handle* handle) {
  if( verbose )
    printf("Executing SSE\n");
  write_handshake(handle);
  write_data(handle, CMD_SSE);
}

void sp_mode_write_memory(struct usb_dev_handle* handle, unsigned long to_addr, const unsigned char* from_here, unsigned long num_words) {
  if( verbose )
    printf("Executing WRAM\n");
  write_handshake(handle);
  write_data(handle, CMD_WRAM|(num_words<<16));
  write_handshake(handle);
  write_data(handle, to_addr);
  while( num_words ) {
//    fputc('.', stdout);
    fflush(stdout);
    write_handshake(handle);
    write_data(handle, ((unsigned long*)from_here)[0]);
    from_here += 4;
    --num_words;
  }
//  printf("\n");
}

unsigned long sp_mode_get_version(struct usb_dev_handle* handle) {
  if( verbose )
    printf("Executing GVE\n");
  write_handshake(handle);
  write_data(handle, CMD_GVE);
  read_handshake(handle);
  return read_data(handle);
}



void do_stuff(struct usb_dev_handle* handle, const char* cmd, unsigned long address, unsigned long count, int fd, unsigned long value, unsigned long delay, unsigned long pause) {
  unsigned char send_buf, recv_buf;
  unsigned long ret, i, chain_length;
  unsigned long flash_data[1024], read_buf[1024], flash_data_len;
  int found;

  int tdi_io = 8;
  int tms_io = 5;
  int tck_io = 1;
  int tdo_io = 3;

  jtag_init(&cmd_buf, handle, tdi_io, tms_io, tck_io, tdo_io, (unsigned char)delay);
  chain_length = get_chain_length(handle);
  printf("Chain length: %d\n", chain_length);

  found = 0;
  for( i = 0; i < chain_length; ++i ) {
    ret = get_device_id(handle, i);
    if( ret == 0x3f0f0f0f )
      found = 1;
    printf("Device %d ID: %08x\n", i, ret);
  }
  if( !found ) {
    printf("No AT91SAM7S chip(s) found.\n");
    jtag_deinit(&cmd_buf);
    return;
  }

  init_serial_programming_mode(handle);
  printf("Serial programming hardware version: %08x\n", sp_mode_get_version(handle));

  if( !strcmp(cmd, "INFO") ) {
    unsigned long lock_bits = sp_mode_get_lock_bits(handle);
    printf("Lock bits: %08x\n", lock_bits);
  } else if( !strcmp(cmd, "ERASE_ALL") ) {
    sp_mode_full_erase(handle);
    printf("Erased.\n");
  } else if( !strcmp(cmd, "READ_FLASH") ) {
    unsigned long* buf = (unsigned long*)malloc(sizeof(unsigned long)*count);
    if( !buf ) {
      fprintf(stderr, "ERROR: Out of memory allocating %d bytes for flash read buffer.\n", sizeof(unsigned long)*count);
      abort();
    }
    sp_mode_read_flash(handle, address, (unsigned char*)buf, count);
    if( fd ) {
      if( write(fd, buf, sizeof(unsigned long)*count) != sizeof(unsigned long)*count ) {
        perror("ERROR: Write failure saving flash read buffer");
        abort();
      }
    } else {
      unsigned long i;
      for( i = 0; i < count; i += 4 ) {
        if( count >= i+4 ) {
          printf("%08x  %08x %08x %08x %08x\n", address+i*16, buf[i], buf[i+1], buf[i+2], buf[i+3]);
        } else if( count >= i+3 ) {
          printf("%08x  %08x %08x %08x\n", address+i*16, buf[i], buf[i+1], buf[i+2]);
        } else if( count >= i+2 ) {
          printf("%08x  %08x %08x\n", address+i*16, buf[i], buf[i+1]);
        } else {
          printf("%08x  %08x\n", address+i*16, buf[i]);
        }
      }
    }
  } else if( !strcmp(cmd, "VERIFY_FLASH") ) {
    unsigned long* buf1 = (unsigned long*)malloc(sizeof(unsigned long)*count);
    unsigned long* buf2 = (unsigned long*)calloc(sizeof(unsigned long), count);
    if( !buf1 ) {
      fprintf(stderr, "ERROR: Out of memory allocating %d bytes for flash read buffer.\n", sizeof(unsigned long)*count);
      abort();
    }
    if( !buf2 ) {
      fprintf(stderr, "ERROR: Out of memory allocating %d bytes for file read buffer.\n", sizeof(unsigned long)*count);
      abort();
    }
    sp_mode_read_flash(handle, address, (unsigned char*)buf1, count);
    read(fd, buf2, sizeof(unsigned long)*count); 
    if( memcmp(buf1, buf2, sizeof(unsigned long)*count) ) {
      printf("Verify failed.\n");
    } else {
      printf("Verify succeeded.\n");
    }
  } else if( !strcmp(cmd, "WRITE_FLASH") ) {
    unsigned long* buf = (unsigned long*)calloc(sizeof(unsigned long), count);
    if( !buf ) {
      fprintf(stderr, "ERROR: Out of memory allocating %d bytes for file read buffer.\n", sizeof(unsigned long)*count);
      abort();
    }
    read(fd, buf, sizeof(unsigned long)*count); 
    sp_mode_write_flash(handle, address, (unsigned char*)buf, count, 1, 0, pause);
    printf("Finished.\n");
  } else if( !strcmp(cmd, "SAFE_WRITE_FLASH") ) {
    unsigned long* buf = (unsigned long*)calloc(sizeof(unsigned long), count);
    if( !buf ) {
      fprintf(stderr, "ERROR: Out of memory allocating %d bytes for file read buffer.\n", sizeof(unsigned long)*count);
      abort();
    }
    read(fd, buf, sizeof(unsigned long)*count); 
    sp_mode_safe_write_flash(handle, address, (unsigned char*)buf, count, 1, 0, pause);
    printf("Finished.\n");
  } else if( !strcmp(cmd, "WRITE_MEM") ) {
    sp_mode_write_memory(handle, address, (unsigned char*)&value, 1);
    printf("Finished.\n");
  }
  jtag_deinit(&cmd_buf);
}

int main(int argc, char** argv) {
  struct usb_bus* bus;
  struct usb_device* dev;
  struct usb_dev_handle* handle;
  int i, num_args, valid_cmd = 0;
  char** args;
  unsigned long address, count, seek, value, delay = 10, pause = 500;
  int fd = 0;

  args = argv+1;
  num_args = argc-1;
  verbose = 0;
  if( num_args > 0 && !strcmp(args[0], "-v") ) {
    verbose = 1;
    ++args;
    --num_args;
  }
  if( num_args > 1 && !strcmp(args[0], "-d") ) {
    char* end;
    delay = strtoul(args[1], &end, 0);
    if( *end || !args[1][0] ) {
      fprintf(stderr, "ERROR: delay must be a valid 32 bit integer.\n");
      return -1;
    }
    if( delay < 0 || delay > 255 ) {
      fprintf(stderr, "ERROR: delay must be between 0 and 255.\n");
      return -1;
    }
    args += 2;
    num_args -= 2;
  }
  if( num_args > 1 && !strcmp(args[0], "-p") ) {
    char* end;
    pause = strtoul(args[1], &end, 0);
    if( *end || !args[1][0] ) {
      fprintf(stderr, "ERROR: pause must be a valid 32 bit integer.\n");
      return -1;
    }
    if( pause < 0 || pause > 1000 ) {
      fprintf(stderr, "ERROR: pause must be between 0 and 1000 (ms).\n");
      return -1;
    }
    args += 2;
    num_args -= 2;
  }

  if( num_args == 1 && (!strcmp(args[0], "INFO") || !strcmp(args[0], "ERASE_ALL")) ) {
    valid_cmd = 1;
  } else if( (num_args == 3 || num_args == 4) && !strcmp(args[0], "READ_FLASH") ) {
    char* end;
    address = strtoul(args[1], &end, 0);
    if( *end || !args[1][0] ) {
      fprintf(stderr, "ERROR: address must be a valid 32 bit integer.\n");
      return -1;
    }
    count = strtoul(args[2], &end, 0);
    if( *end || !args[1][0] ) {
      fprintf(stderr, "ERROR: count must be a valid 32 bit integer.\n");
      return -1;
    }
    if( num_args == 4 ) {
      fd = open(args[3], O_WRONLY|O_CREAT|O_TRUNC, 0755);
      if( !fd ) {
        perror("ERROR: Unable to create file to write flash data to");
        return -2;
      }
    }
    valid_cmd = 1;
  } else if( (num_args == 4 || num_args == 5) && (!strcmp(args[0], "WRITE_FLASH") || !strcmp(args[0], "SAFE_WRITE_FLASH") || !strcmp(args[0], "VERIFY_FLASH")) ) {
    char* end;
    address = strtoul(args[1], &end, 0);
    if( *end || !args[1][0] ) {
      fprintf(stderr, "ERROR: (address) must be a valid 32 bit integer.\n");
      return -1;
    }
    if( num_args == 5 ) {
      count = strtoul(args[2], &end, 0);
      if( *end || !args[2][0] ) {
        fprintf(stderr, "ERROR: count must be a valid 32 bit integer.\n");
        return -1;
      }
    }
    fd = open(args[num_args == 5 ? 3 : 2], O_RDONLY);
    if( !fd ) {
      perror("ERROR: Unable to open file to read flash data from");
      return -2;
    }
    seek = strtoul(args[num_args == 5 ? 4 : 3], &end, 0);
    if( *end || !args[num_args == 5 ? 4 : 3][0] ) {
      fprintf(stderr, "ERROR: seek must be a valid 32 bit integer.\n");
      return -1;
    }
    if( num_args == 4 ) {
      count = lseek(fd, 0, SEEK_END)>>2;
      lseek(fd, 0, SEEK_SET);
      count = ((count+63)>>6)<<6; // round up to nearest # of pages
    }
    if( seek )
      lseek(fd, seek, SEEK_SET);
    valid_cmd = 1;
  } else if( num_args == 3 && !strcmp(args[0], "WRITE_MEM") ) {
    char* end;
    address = strtoul(args[1], &end, 0);
    if( *end || !args[1][0] ) {
      fprintf(stderr, "ERROR: (address) must be a valid 32 bit integer.\n");
      return -1;
    }
    value = strtoul(args[2], &end, 0);
    if( *end || !args[2][0] ) {
      fprintf(stderr, "ERROR: value must be a valid 32 bit integer.\n");
      return -1;
    }
    valid_cmd = 1;
  }

  if( !valid_cmd ) {
    int len = strlen(argv[0]);
    printf("Usage: %.*s                            HELP\n"
           "       %.*s [-d delay] [-p pause] [-v] INFO\n"
           "       %.*s [-d delay] [-p pause] [-v] ERASE_ALL\n"
           "       %.*s [-d delay] [-p pause] [-v] READ_FLASH       <addr>  <length>  [<file>]\n"
           "       %.*s [-d delay] [-p pause] [-v] WRITE_FLASH      <addr> [<length>]  <file> <seek> [ERASE] [LOCK]\n"
           "       %.*s [-d delay] [-p pause] [-v] SAFE_WRITE_FLASH <addr> [<length>]  <file> <seek> [ERASE] [LOCK]\n"
           "       %.*s [-d delay] [-p pause] [-v] VERIFY_FLASH     <addr> [<length>]  <file> <seek>\n"
           "       %.*s [-d delay] [-p pause] [-v] WRITE_MEM        <addr>  <value>\n",
           len, argv[0],
           len, argv[0],
           len, argv[0],
           len, argv[0],
           len, argv[0]);
    return 0;
  }

  usb_init();
  usb_find_busses();
  usb_find_devices();
  bus = &usb_busses[0];
  while(bus) {
    dev = &bus->devices[0];
    while(dev) {
      if( dev->descriptor.idVendor == 0x6667 && dev->descriptor.idProduct == 0x6663 ) {
        handle = usb_open(dev);
        if( handle ) {
          usb_claim_interface(handle, 0);
          do_stuff(handle, args[0], address, count, fd, value, delay, pause);
          usb_close(handle);
          if( fd )
            close(fd);
          return 0;
        } else {
          fprintf(stderr, "Error: Found device, but open command failed.\n");
          return -1;
        }
      }
      dev = dev->next;
    }
    bus = bus->next;
  }
  if( fd )
    close(fd);
  fprintf(stderr, "Error: Can't find USB device (0x1781:0x0c9e).\n");
  return -2;
}
